﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.ServicesWrapper.EIS;
using BMS.Facade.Data;
using BMS.Facade;
using VI = BMS.VistaIntegration.FacadeContracts;
using BMS.Utils;
using DC = BMS.DataContracts;
using BMS.ServicesWrapper.BMService;
using System.ServiceModel;
using System.Threading;
using System.Diagnostics;
using System.Reflection;
using System.IO;
using BMS.VistaWorker2.Writer;
using BMS.VistaWorker2.Writer.Implementation;
using BMS.VistaIntegration.Data;
using BMS.VistaIntegration.VistA;
using InfoWorld.HL7.ITS;

namespace Import_VistaJob
{
    class Program
    {
        [Flags]
        enum ExitCodes : int
        {
            Success = 0,
            PartialSuccess = 1,
            InvalidParameter = 2,
            Fail = 4,
            UnknownError = 8,
            JobStarted = 16
        }

        static ExitCodes exitCode;

        public static void Main(string[] args)
        {
            try
            {
                string vistaSiteCode = null, ip = null, methodType = null, port = null, user = null, password = null, database = null, schema = null, dataTypes = null;
                VistaSite vista = null;
                TimeZoneInfo tzi = null;
                DateTime startDate; DateTime endDate;
                VI.VistASite vistaSettings = null;
                Dictionary<string, string> parameters = new Dictionary<string, string>();
                string test = string.Empty;
                //find vista site
                foreach (string cmdParameter in args)
                {
                    if (cmdParameter.StartsWith("/") && cmdParameter.Contains('='))
                    {
                        string[] commandParameter = cmdParameter.Split('=');
                        string parameterName = commandParameter[0].Substring(1, commandParameter[0].Length - 1);
                        string parameterValue = commandParameter[1];
                        if (!parameters.ContainsKey(parameterName))
                            parameters.Add(parameterName, parameterValue);
                        else
                        {
                            Console.Write(string.Format("{0} {1}", "Duplicate parameter:", parameterName));
                            System.Environment.Exit((int)ExitCodes.InvalidParameter);
                        }
                    }
                }

                if (parameters.ContainsKey("VistaSiteCode"))
                {
                    vistaSiteCode = parameters["VistaSiteCode"];
                    if ((string.IsNullOrEmpty(vistaSiteCode) || vistaSiteCode.Length != 6))
                    {
                        Console.WriteLine("The parameter value provided for VistaSiteCode is incorrect!");
                        System.Environment.Exit((int)ExitCodes.InvalidParameter);
                    }
                    else
                    {
                        vista = EISFactory.InstanceWindows.GetVistaSiteByCode(vistaSiteCode);
                        if (vista == null)
                        {
                            Console.WriteLine("Vista Site not found!");
                            System.Environment.Exit((int)ExitCodes.Fail);
                        }
                        List<VI.VistASite> vistaSites = FacadeManager.ConfigurationInterface.GetVistaSites(Utils.DomainId).ToList();
                        vistaSettings = vistaSites.Where(a => a.Id.Equals(vista.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        string theDirectory = Path.GetDirectoryName(new Uri(Assembly.GetExecutingAssembly().CodeBase).LocalPath);
                        Trace.Listeners.Add(new TextWriterTraceListener(theDirectory + "\\Import_VistaJob." + vistaSiteCode + ".log", "myListener"));
                    }
                }
                else
                {
                    Console.Write("The parameter VistaSiteCode is missing!");
                    System.Environment.Exit((int)ExitCodes.InvalidParameter);
                }

                if (vistaSettings == null)
                {
                    tzi = GetTimeZone(parameters);
                }

                if (parameters.ContainsKey("QueryMethod"))
                {
                    methodType = parameters["QueryMethod"];
                    if (methodType == "MDWS")
                    {
                        if (vistaSettings == null || !string.IsNullOrEmpty(TestConnection(vistaSettings, DataRetrievalMethod.MDWS)))
                        {
                            vistaSettings = new VI.VistASite();
                            vistaSettings.Id = vista.Id.extension;
                            vistaSettings.Name = vista.Code;
                            if (tzi == null)
                                tzi = GetTimeZone(parameters);
                            vistaSettings.TimeZone = tzi;
                            vistaSettings.Number = vista.Number;
                            try
                            {
                                test = TestConnection(vistaSettings, DataRetrievalMethod.MDWS);
                                if (!string.IsNullOrEmpty(test))
                                {
                                    Console.WriteLine("ERROR: MDWS connection - " + test);
                                    System.Environment.Exit((int)ExitCodes.Fail);
                                }
                                else
                                    FacadeManager.ConfigurationInterface.SaveVistASite(vistaSettings, Utils.DomainId);
                            }
                            catch (Exception ex)
                            {
                                Console.WriteLine("ERROR: MDWS connection - " + ex.Message);
                                System.Environment.Exit((int)ExitCodes.Fail);
                            }
                        }
                    }
                    else if (methodType == "ODBC")
                    {
                        if (vistaSettings == null || !string.IsNullOrEmpty(TestConnection(vistaSettings, DataRetrievalMethod.ODBC)))
                        {
                            vistaSettings = new VI.VistASite();
                            vistaSettings.Id = vista.Id.extension;
                            vistaSettings.Name = vista.Code;
                            if (tzi == null)
                                tzi = GetTimeZone(parameters);
                            vistaSettings.TimeZone = tzi;
                            vistaSettings.Number = vista.Number;

                            if (parameters.ContainsKey("VistaIP"))
                            {
                                ip = parameters["VistaIP"];
                            }
                            else
                            {
                                Console.Write("The parameter VistaIP is missing!");
                                System.Environment.Exit((int)ExitCodes.InvalidParameter);
                            }

                            if (parameters.ContainsKey("VistaPort"))
                            {
                                port = parameters["VistaPort"];
                            }
                            else
                            {
                                Console.Write("The parameter VistaPort is missing!");
                                System.Environment.Exit((int)ExitCodes.InvalidParameter);
                            }

                            if (parameters.ContainsKey("ODBCUser"))
                            {
                                user = parameters["ODBCUser"];
                            }
                            else
                            {
                                Console.Write("The parameter ODBCUser is missing!");
                                System.Environment.Exit((int)ExitCodes.InvalidParameter);
                            }

                            if (parameters.ContainsKey("ODBCPassword"))
                            {
                                password = parameters["ODBCPassword"];
                            }
                            else
                            {
                                Console.Write("The parameter ODBCPassword is missing!");
                                System.Environment.Exit((int)ExitCodes.InvalidParameter);
                            }

                            if (parameters.ContainsKey("ODBCDatabase"))
                            {
                                database = parameters["ODBCDatabase"];
                            }
                            else
                            {
                                Console.Write("The parameter ODBCDatabase is missing!");
                                System.Environment.Exit((int)ExitCodes.InvalidParameter);
                            }

                            if (parameters.ContainsKey("ODBCSchema"))
                            {
                                schema = parameters["ODBCSchema"];
                            }
                            else
                            {
                                Console.Write("The parameter ODBCSchema is missing!");
                                System.Environment.Exit((int)ExitCodes.InvalidParameter);
                            }

                            vistaSettings.OdbcUser = user;
                            vistaSettings.OdbcPassword = password;
                            vistaSettings.OdbcConnectionString = "Server=" + ip + ";Port=" + port + ";Database=" + database + ";Schema=" + schema + ";";
                            try
                            {
                                test = TestConnection(vistaSettings, DataRetrievalMethod.ODBC);
                                if (!string.IsNullOrEmpty(test))
                                {
                                    Console.WriteLine("ERROR: ODBC connection - " + test);
                                    System.Environment.Exit((int)ExitCodes.Fail);
                                }
                                else
                                    FacadeManager.ConfigurationInterface.SaveVistASite(vistaSettings, Utils.DomainId);
                            }
                            catch (Exception ex)
                            {
                                Console.WriteLine("ERROR: ODBC connection - " + ex.Message);
                                System.Environment.Exit((int)ExitCodes.Fail);
                            }
                        }
                    }
                    else
                    {
                        Console.WriteLine("The parameter value provided for QueryMethod is incorrect!");
                        System.Environment.Exit((int)ExitCodes.InvalidParameter);
                    }
                }
                else
                {
                    Console.WriteLine("The parameter QueryMethod is missing!");
                    System.Environment.Exit((int)ExitCodes.InvalidParameter);
                }

                if (parameters.ContainsKey("DataType"))
                {
                    dataTypes = parameters["DataType"];
                    switch (dataTypes)
                    {
                        case "ADT":
                            if (parameters.ContainsKey("StartDate"))
                            {
                                if (string.IsNullOrEmpty(parameters["StartDate"]) || !DateTime.TryParse(parameters["StartDate"], out startDate))
                                {
                                    Console.WriteLine("The parameter value provided for StartDate is incorrect!");
                                    System.Environment.Exit((int)ExitCodes.InvalidParameter);
                                }
                                else
                                {
                                    if (parameters.ContainsKey("EndDate"))
                                    {
                                        if (string.IsNullOrEmpty(parameters["EndDate"]) || !DateTime.TryParse(parameters["EndDate"], out endDate))
                                        {
                                            Console.WriteLine("The parameter value provided for EndDate is incorrect!");
                                            System.Environment.Exit((int)ExitCodes.InvalidParameter);
                                        }
                                        else
                                        {
                                            if (string.IsNullOrEmpty(test))
                                                ADTProcessor.ImportADTData(vistaSettings, methodType, startDate, endDate);
                                        }
                                    }
                                    else
                                        Console.Write("The parameter EndDate is missing!");
                                }
                            }
                            else
                                Console.Write("The parameter StartDate is missing!");
                            break;
                        case "Specialty":
                            if (string.IsNullOrEmpty(test))
                                ImportSpecialtyData(vistaSettings, methodType);
                            break;
                        case "TreatingSpecialty":
                            if (string.IsNullOrEmpty(test))
                                ImportTreatingSpecialtyData(vistaSettings, methodType);
                            break;
                        case "FacilityMovementType":
                            if (string.IsNullOrEmpty(test))
                                ImportFacilityMovementTypeData(vistaSettings, methodType);
                            break;
                        case "OrderableItem":
                            if (string.IsNullOrEmpty(test))
                                ImportOrderableItemData(vistaSettings, methodType);
                            break;
                        case "HospitalLocation":
                            if (string.IsNullOrEmpty(test))
                                HospitalLocationProcessor.ImportHospitalLocationData(vistaSettings, methodType);
                            break;
                        case "MedicalCenterDivision":
                            if (string.IsNullOrEmpty(test))
                                ImportMedicalCenterDivisionData(vistaSettings, methodType);
                            break;
                        case "WardLocation":
                            if (string.IsNullOrEmpty(test))
                                ImportWardLocationData(vistaSettings, methodType);
                            break;
                        case "RoomBed":
                            if (string.IsNullOrEmpty(test))
                                ImportRoomBedData(vistaSettings, methodType);
                            break;
                        case "Patient":
                            if (string.IsNullOrEmpty(test))
                                PatientProcessor.ImportPatientsData(vistaSettings, methodType);
                            break;
                        case "UpdateADTBeds":
                            if (string.IsNullOrEmpty(test))
                                UpdateADTBeds(vistaSettings, methodType, vista);
                            break;
                        default:
                            Console.WriteLine("The parameter value provided for DataType is incorrect!");
                            break;
                    }
                }
                else
                {
                    Console.Write("The parameter DataType is missing!");
                    System.Environment.Exit((int)ExitCodes.InvalidParameter);

                }
                System.Environment.Exit((int)exitCode);
            }
            catch(Exception ex)
            {
                Console.WriteLine("Error: " + ex.Message);
                Tracer.TraceMessage("Error: " + ex.Message);
                System.Environment.Exit((int)ExitCodes.Fail);
            }
        }

        private static void ImportSpecialtyData(VI.VistASite vistaSettings, string methodType)
        {
            DataRetrievalMethod? retrievalMethod = null;
            bool _isSuccess = true;
            if (methodType.Equals(DataRetrievalMethod.MDWS.ToString(), StringComparison.InvariantCultureIgnoreCase))
                retrievalMethod = DataRetrievalMethod.MDWS;
            else if (methodType.Equals(DataRetrievalMethod.ODBC.ToString(), StringComparison.InvariantCultureIgnoreCase))
                retrievalMethod = DataRetrievalMethod.ODBC;

            DC.VistASite vistaSite = Utils.GetVistaSite(vistaSettings, retrievalMethod.Value);
            try
            {
                BMSFactory.VistaQueryClientWindows.RunOnDemand(vistaSite, VistaDataType.Specialty, DateTime.UtcNow.AddYears(-100), DateTime.UtcNow.AddYears(10), retrievalMethod.Value);
            }
            catch (FaultException<BMS.FaultContracts.VistAException> e)
            {
                _isSuccess = false;
                Console.WriteLine("ERROR: " + e.Detail.ErrorMessage);
                exitCode = ExitCodes.Fail;
                return;
            }
            if (_isSuccess)
            {
                Console.WriteLine("Vista Job started! Go to the Audit page to watch the progress.");
                exitCode = ExitCodes.JobStarted;
            }
            Utils.InsertVistaOperation(vistaSettings.Id, "32", DateTime.UtcNow);
        }

        private static void ImportTreatingSpecialtyData(VI.VistASite vistaSettings, string methodType)
        {
            DataRetrievalMethod? retrievalMethod = null;
            bool _isSuccess = true;
            if (methodType.Equals(DataRetrievalMethod.MDWS.ToString(), StringComparison.InvariantCultureIgnoreCase))
                retrievalMethod = DataRetrievalMethod.MDWS;
            else if (methodType.Equals(DataRetrievalMethod.ODBC.ToString(), StringComparison.InvariantCultureIgnoreCase))
                retrievalMethod = DataRetrievalMethod.ODBC;

            DC.VistASite vistaSite = Utils.GetVistaSite(vistaSettings, retrievalMethod.Value);
            try
            {
                BMSFactory.VistaQueryClientWindows.RunOnDemand(vistaSite, VistaDataType.TreatingSpecialty, DateTime.UtcNow.AddYears(-100), DateTime.UtcNow.AddYears(10), retrievalMethod.Value);
            }
            catch (FaultException<BMS.FaultContracts.VistAException> e)
            {
                _isSuccess = false;
                Console.WriteLine("ERROR: " + e.Detail.ErrorMessage);
                exitCode = ExitCodes.Fail;
                return;
            }
            if (_isSuccess)
            {
                Console.WriteLine("Vista Job started! Go to the Audit page to watch the progress.");
                exitCode = ExitCodes.JobStarted;
            }
            Utils.InsertVistaOperation(vistaSettings.Id, "64", DateTime.UtcNow);
        }

        private static void ImportFacilityMovementTypeData(VI.VistASite vistaSettings, string methodType)
        {
            DataRetrievalMethod? retrievalMethod = null;
            bool _isSuccess = true;
            if (methodType.Equals(DataRetrievalMethod.MDWS.ToString(), StringComparison.InvariantCultureIgnoreCase))
                retrievalMethod = DataRetrievalMethod.MDWS;
            else if (methodType.Equals(DataRetrievalMethod.ODBC.ToString(), StringComparison.InvariantCultureIgnoreCase))
                retrievalMethod = DataRetrievalMethod.ODBC;

            DC.VistASite vistaSite = Utils.GetVistaSite(vistaSettings, retrievalMethod.Value);
            try
            {
                BMSFactory.VistaQueryClientWindows.RunOnDemand(vistaSite, VistaDataType.FacilityMovementType, DateTime.UtcNow.AddYears(-100), DateTime.UtcNow.AddYears(10), retrievalMethod.Value);
            }
            catch (FaultException<BMS.FaultContracts.VistAException> e)
            {
                _isSuccess = false;
                Tracer.TraceException(e);
                Console.WriteLine("ERROR: " + e.Detail.ErrorMessage);
                exitCode = ExitCodes.Fail;
                return;
            }
            if (_isSuccess)
            {
                Console.WriteLine("Vista Job started! Go to the Audit page to watch the progress.");
                exitCode = ExitCodes.JobStarted;
            }
            Utils.InsertVistaOperation(vistaSettings.Id, "128", DateTime.UtcNow);

        }

        private static void ImportOrderableItemData(VI.VistASite vistaSettings, string methodType)
        {
            DataRetrievalMethod? retrievalMethod = null;
            bool _isSuccess = true;
            if (methodType.Equals(DataRetrievalMethod.MDWS.ToString(), StringComparison.InvariantCultureIgnoreCase))
                retrievalMethod = DataRetrievalMethod.MDWS;
            else if (methodType.Equals(DataRetrievalMethod.ODBC.ToString(), StringComparison.InvariantCultureIgnoreCase))
                retrievalMethod = DataRetrievalMethod.ODBC;

            DC.VistASite vistaSite = Utils.GetVistaSite(vistaSettings, retrievalMethod.Value);
            try
            {
                BMSFactory.VistaQueryClientWindows.RunOnDemand(vistaSite, VistaDataType.OrderableItem, DateTime.UtcNow.AddYears(-100), DateTime.UtcNow.AddYears(10), retrievalMethod.Value);
            }
            catch (FaultException<BMS.FaultContracts.VistAException> e)
            {
                _isSuccess = false;
                Tracer.TraceException(e);
                Console.WriteLine("ERROR: " + e.Detail.ErrorMessage);
                exitCode = ExitCodes.Fail;
                return;
            }
            if (_isSuccess)
            {
                Console.WriteLine("Vista Job started! Go to the Audit page to watch the progress.");
                exitCode = ExitCodes.JobStarted;
            }
            Utils.InsertVistaOperation(vistaSettings.Id, "256", DateTime.UtcNow);
        }

        private static void ImportMedicalCenterDivisionData(VI.VistASite vistaSettings, string methodType)
        {
            DataRetrievalMethod? retrievalMethod = null;
            bool _isSuccess = true;
            if (methodType.Equals(DataRetrievalMethod.MDWS.ToString(), StringComparison.InvariantCultureIgnoreCase))
                retrievalMethod = DataRetrievalMethod.MDWS;
            else if (methodType.Equals(DataRetrievalMethod.ODBC.ToString(), StringComparison.InvariantCultureIgnoreCase))
                retrievalMethod = DataRetrievalMethod.ODBC;

            DC.VistASite vistaSite = Utils.GetVistaSite(vistaSettings, retrievalMethod.Value);
            try
            {
                BMSFactory.VistaQueryClientWindows.RunOnDemand(vistaSite, VistaDataType.MedicalCenterDivision, DateTime.UtcNow.AddYears(-100), DateTime.UtcNow.AddYears(10), retrievalMethod.Value);
            }
            catch (FaultException<BMS.FaultContracts.VistAException> e)
            {
                _isSuccess = false;
                Tracer.TraceException(e);
                Console.WriteLine("ERROR: " + e.Detail.ErrorMessage);
                exitCode = ExitCodes.Fail;
                return;
            }
            if (_isSuccess)
            {
                Console.WriteLine("Vista Job started! Go to the Audit page to watch the progress.");
                exitCode = ExitCodes.JobStarted;
            }
            Utils.InsertVistaOperation(vistaSettings.Id, "512", DateTime.UtcNow);
        }

        private static void ImportWardLocationData(VI.VistASite vistaSettings, string methodType)
        {
            DataRetrievalMethod? retrievalMethod = null;
            bool _isSuccess = true;
            if (methodType.Equals(DataRetrievalMethod.MDWS.ToString(), StringComparison.InvariantCultureIgnoreCase))
                retrievalMethod = DataRetrievalMethod.MDWS;
            else if (methodType.Equals(DataRetrievalMethod.ODBC.ToString(), StringComparison.InvariantCultureIgnoreCase))
                retrievalMethod = DataRetrievalMethod.ODBC;

            DC.VistASite vistaSite = Utils.GetVistaSite(vistaSettings, retrievalMethod.Value);
            try
            {
                BMSFactory.VistaQueryClientWindows.RunOnDemand(vistaSite, VistaDataType.WardLocation, DateTime.UtcNow.AddYears(-100), DateTime.UtcNow.AddYears(10), retrievalMethod.Value);
            }
            catch (FaultException<BMS.FaultContracts.VistAException> e)
            {
                _isSuccess = false;
                Tracer.TraceException(e);
                Console.WriteLine("ERROR: " + e.Detail.ErrorMessage);
                exitCode = ExitCodes.Fail;
                return;
            }
            if (_isSuccess)
            {
                Console.WriteLine("Vista Job started! Go to the Audit page to watch the progress.");
                exitCode = ExitCodes.JobStarted;
            }
            Utils.InsertVistaOperation(vistaSettings.Id, "8", DateTime.UtcNow);
        }

        private static void ImportRoomBedData(VI.VistASite vistaSettings, string methodType)
        {
            DataRetrievalMethod? retrievalMethod = null;
            bool _isSuccess = true;
            if (methodType.Equals(DataRetrievalMethod.MDWS.ToString(), StringComparison.InvariantCultureIgnoreCase))
                retrievalMethod = DataRetrievalMethod.MDWS;
            else if (methodType.Equals(DataRetrievalMethod.ODBC.ToString(), StringComparison.InvariantCultureIgnoreCase))
                retrievalMethod = DataRetrievalMethod.ODBC;

            DC.VistASite vistaSite = Utils.GetVistaSite(vistaSettings, retrievalMethod.Value);
            try
            {
                BMSFactory.VistaQueryClientWindows.RunOnDemand(vistaSite, VistaDataType.RoomBed, DateTime.UtcNow.AddYears(-100), DateTime.UtcNow.AddYears(10), retrievalMethod.Value);
            }
            catch (FaultException<BMS.FaultContracts.VistAException> e)
            {
                _isSuccess = false;
                Tracer.TraceException(e);
                Console.WriteLine("ERROR: " + e.Detail.ErrorMessage);
                exitCode = ExitCodes.Fail;
                return;
            }
            if (_isSuccess)
            {
                Console.WriteLine("Vista Job started! Go to the Audit page to watch the progress.");
                exitCode = ExitCodes.JobStarted;
            }
            Utils.InsertVistaOperation(vistaSettings.Id, "4", DateTime.UtcNow);
        }

        private static void UpdateADTBeds(VI.VistASite vistaSettings, string methodType, VistaSite vista)
        {
            DataRetrievalMethod? retrievalMethod = null;
            bool _isSuccess = true;
            if (methodType.Equals(DataRetrievalMethod.MDWS.ToString(), StringComparison.InvariantCultureIgnoreCase))
                retrievalMethod = DataRetrievalMethod.MDWS;
            else if (methodType.Equals(DataRetrievalMethod.ODBC.ToString(), StringComparison.InvariantCultureIgnoreCase))
                retrievalMethod = DataRetrievalMethod.ODBC;

            DC.VistASite vistaSite = Utils.GetVistaSite(vistaSettings, retrievalMethod.Value);
            try
            {
                IWriterManager writerManager = new WriterManagerFactory().MakeWriter();
                writerManager.Open(new VistASite(vistaSettings.Id, vistaSettings.Name, vistaSettings.Number, vistaSettings.TimeZone, null));
                IVistAQuery query = Utils.GetVistASession(vistaSettings, retrievalMethod.Value).MakeQuery();

                List<OccupiedBedInfo> list = FacadeManager.BedInterface.GetOccupiedBeds(vista.Id).ToList();
                Dictionary<string, OccupiedBedInfo> bmsDictionaryBedSwitch = new Dictionary<string, OccupiedBedInfo>();
                foreach (OccupiedBedInfo obi in list)
                {
                    if (!bmsDictionaryBedSwitch.ContainsKey(obi.ActIEN))
                        bmsDictionaryBedSwitch.Add(obi.ActIEN, obi);
                }

                if (!bmsDictionaryBedSwitch.Keys.Any()) return;
                List<BMS.VistaIntegration.Data.WF.BedSwitch> vistABedSwitch = query.GetBedsSwitch(bmsDictionaryBedSwitch.Keys).ToList();
                vistABedSwitch.RemoveAll(a => string.IsNullOrEmpty(a.WardIen) || string.IsNullOrEmpty(a.RoomBedIen));
                var bedSwitchChanged = from b in vistABedSwitch
                                       let bmsBedSwitch = bmsDictionaryBedSwitch[b.Ien]
                                       where (b.RoomBedIen != bmsBedSwitch.BedIEN)
                                       select new { bmsBedSwitch, b };
                
                bedSwitchChanged.ForEach(s =>
                {
                    ChangePatientBedSwitch(s.bmsBedSwitch, s.b, vista.Id);
                });
            }
            catch (Exception ex)
            {
                _isSuccess = false;
                Tracer.TraceException(ex);
                Console.WriteLine("ERROR: " + ex.Message);
                exitCode = ExitCodes.Fail;
                return;
            }
            if (_isSuccess)
            {
                Console.WriteLine("Finished updating beds for movements.");
                exitCode = ExitCodes.Success;
            }
        }

        private static void ChangePatientBedSwitch(OccupiedBedInfo bmsOccupiedBed, BMS.VistaIntegration.Data.WF.BedSwitch vistaBedInfo, II vistaSiteId)
        {                        
            try
            {
                string domain = BMS.ServicesWrapper.Security.SecurityFactory.InstanceWindows.GetCurrentDomain();
                Bed bed = EISFactory.InstanceWindows.GetBed(vistaBedInfo.RoomBedIen, vistaSiteId);
                Bed oldBed = EISFactory.InstanceWindows.GetBed(bmsOccupiedBed.BedIEN, vistaSiteId);

                if (bed != null && oldBed != null && bed.Name.Equals(oldBed.Name, StringComparison.InvariantCultureIgnoreCase))
                {
                    Ward ward = EISFactory.InstanceWindows.GetWard(vistaBedInfo.WardIen, vistaSiteId);
                    if (ward != null && bed != null)
                    {
                        BMS.Facade.Data.Patient patient = EISFactory.InstanceWindows.GetPatientBySsn(new II(Constants.SSNROOT, bmsOccupiedBed.PatientSSN), null);
                        BMS.DataContracts.AdmissionEvent adm = null;
                        if (bmsOccupiedBed.IsAdmission)
                            adm = BMSFactory.BedManagerOperationsClientWindows.GetAdmissionEvent(bmsOccupiedBed.Id);
                        else
                        {
                            BMS.DataContracts.MovementEvent mov = BMSFactory.BedManagerOperationsClientWindows.GetMovementEvent(bmsOccupiedBed.Id);
                            adm = BMSFactory.BedManagerOperationsClientWindows.GetAdmissionEvent(mov.AdmissionId);
                            mov.WardId = ward.Id;
                            mov.BedId = bed.Id;
                            BMSFactory.BedManagerOperationsClientWindows.UpdateMovementEvent(mov);
                        }
                        adm.WardId = ward.Id;
                        adm.BedId = bed.Id;
                        BMSFactory.BedManagerOperationsClientWindows.UpdateAdmissionEvent(adm);
                        Tracer.TraceMessage("Bed switch - update movement for patient ssn " + bmsOccupiedBed.PatientSSN);                                                
                    }
                }
            }
            catch (Exception e)
            {
                Tracer.TraceMessage("Exception processing bed switch for patient: " + bmsOccupiedBed.PatientSSN);
                Tracer.TraceException(e);
            }
        }

        private static TimeZoneInfo GetTimeZone(Dictionary<string, string> parameters)
        {
            string timezone = null;
            TimeZoneInfo tzi = null;
            if (!parameters.ContainsKey("VistaTimeZone"))
            {
                Console.Write("The parameter VistaTimeZone is missing!");
                System.Environment.Exit((int)ExitCodes.InvalidParameter);
            }
            else
            {
                switch (parameters["VistaTimeZone"])
                {
                    case "EST":
                        timezone = "Eastern Standard Time";
                        break;
                    case "CST":
                        timezone = "Central Standard Time";
                        break;
                    case "MST":
                        timezone = "Mountain Standard Time";
                        break;
                    case "PST":
                        timezone = "Pacific Standard Time";
                        break;
                    case "HST":
                        timezone = "Hawaiian Standard Time";
                        break;
                    case "AST":
                        timezone = "Atlantic Standard Time";
                        break;
                    case "AKST":
                        timezone = "Alaskan Standard Time";
                        break;
                    case "EEST":
                        timezone = "GTB Standard Time";
                        break;
                    default:
                        timezone = string.Empty;
                        break;
                }
                if (string.IsNullOrEmpty(timezone) || timezone.Length < 3)
                {
                    Console.WriteLine("The parameter value provided for VistaTimeZone is incorrect!");
                    System.Environment.Exit((int)ExitCodes.InvalidParameter);
                }
                else
                {
                    try
                    {
                        tzi = TimeZoneInfo.FindSystemTimeZoneById(timezone);
                    }
                    catch (Exception ex)
                    {
                        Tracer.TraceException(ex);
                        Console.WriteLine(ex.Message);
                        System.Environment.Exit((int)ExitCodes.Fail);
                    }
                }
            }
            return tzi;
        }

        private static string TestConnection(VI.VistASite site, DataRetrievalMethod retrievalMethod)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DC.VistASite vistaSite = new DC.VistASite();
                vistaSite.VistaName = site.Name;
                vistaSite.TimeZoneId = site.TimeZone.Id;
                vistaSite.Id = site.Id;
                vistaSite.Number = site.Number;
                if (retrievalMethod == DataRetrievalMethod.ODBC)
                {
                    vistaSite.AccessCode = site.OdbcUser;
                    vistaSite.VerifyCode = site.OdbcPassword;
                    vistaSite.ConnectionString = site.OdbcConnectionString;
                }                
                try
                {
                    BMSFactory.VistaQueryClientWindows.TestConnection(vistaSite, retrievalMethod);                    
                }
                catch (FaultException<BMS.FaultContracts.VistAException> ce) { return ce.Detail.ErrorMessage; }
                return string.Empty;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
